{*******************************************************}
{                                                       }
{     StreamSec Security Library for Embarcadero Delphi }
{     smTLSIn10IOHandler Unit                           }
{                                                       }
{     For Indy 10 Delphi 13                             }
{                                                       }
{     Copyright (C) 2003-2025 StreamSec Handelsbolag    }
{                                                       }
{*******************************************************}
unit smTLSId10D37IOHandler;

interface

uses
  SysUtils, Classes,
  StreamSec.Mobile.TlsInternalServer,
  IdStackConsts, IdGlobal, IdStack,
  IdThread, IdSocketHandle, IdBaseComponent, IdComponent, IdIOHandler,
  IdIOHandlerSocket,
  IdServerIOHandler, IdSSL, IdYarn,
  StreamSec.Mobile.StreamSecII;

type
  TBytes = TIdBytes;

  // TLS wrapper classes:
  TCustomTLSIdSock = class(TAbstractTLSSocket)
  private
    FSocket: TIdSocketHandle;
    FPort: Integer;
    FHost: string;
    FService: string;
    FAddress: string;
    FRemoteHost: string;
    FRemoteAddress: string;
    FRemotePort: Integer;
    FBindMaxPort: Integer;
    FBindMinPort: Integer;
    FBindPort: Integer;
    FBindAddress: string;
    FTimeOut: Cardinal;
    procedure SetAddress(const Value: string);
    procedure SetHost(const Value: string);
    procedure SetPort(const Value: Integer);
    procedure SetService(const Value: string);
    function GetRemoteHost: string;
    function GetRemoteAddress: string;
    function GetRemotePort: Integer;
    procedure SetBindAddress(const Value: string);
    procedure SetBindMaxPort(const Value: Integer);
    procedure SetBindMinPort(const Value: Integer);
    procedure SetBindPort(const Value: Integer);
    procedure SetTimeOut(const Value: Cardinal);
  protected
    procedure DoRead; virtual;
    procedure RawClose; override;
    procedure RawConnect; override;
    function RawReceive: TStream; override;
    procedure RawSend(Strm: TCustomMemoryStream); override;
  public
    constructor Create(ASocket: TIdSocketHandle); virtual;
    property Address: string read FAddress write SetAddress;
    property Host: string read FHost write SetHost;
    property Port: Integer read FPort write SetPort;
    property Service: string read FService write SetService;
    property BindAddress: string read FBindAddress write SetBindAddress;
    property BindPort: Integer read FBindPort write SetBindPort;
    property BindMinPort: Integer read FBindMinPort write SetBindMinPort;
    property BindMaxPort: Integer read FBindMaxPort write SetBindMaxPort;
    property RemoteHost: string read GetRemoteHost;
    property RemoteAddress: string read GetRemoteAddress;
    property RemotePort: Integer read GetRemotePort;
    property TimeOut: Cardinal read FTimeOut write SetTimeOut;
    property Version;
    property SelectedCipherSuite;
  end;

  TNewSynConnectEvent = procedure (Sender: TObject; Socket: TCustomTLSIdSock) of object;

  TTLSIdSockSlave = class(TCustomTLSIdSock)
  private
    FSlaveSocket: TIdSocketHandle;
  protected
    procedure Connect; override;
    procedure Disconnect; override;
    procedure Receive; override;
    procedure Send; override;
    procedure DoConnect; virtual;
  public
    constructor Create(ASocket: TIdSocketHandle); override;
    constructor CreateSocket(ASocket: TIdStackSocketHandle);
    destructor Destroy; override;
  end;

  TBlockTLSIdSock = class(TTLSIdSockSlave)
  private
    FOnConnect: TNotifyEvent;
    procedure SetOnConnect(const Value: TNotifyEvent);
  protected
    procedure DoConnect; override;
  public
    property OnConnect: TNotifyEvent read FOnConnect write SetOnConnect;
  end;

  // TIdSocketStream is used internally as a substitute for
  // ScktComp.TWinSocketStream
  TInSocketStream = class(TStream)
  private
    FSocket: TIdSocketHandle;
    FTimeout: Longint;
    FReadable: Boolean;
  public
    constructor Create(ASocket: TIdSocketHandle; TimeOut: Longint);
    function WaitForData(Timeout: Longint): Boolean;
    function Read(var Buffer; Count: Longint): Longint; override;
    function Write(const Buffer; Count: Longint): Longint; override;
    function Seek(Offset: Longint; Origin: Word): Longint; override;
    property TimeOut: Longint read FTimeout write FTimeout;
  end;

  TsmTLSIdIOHandlerSocket = class(TIdSSLIOHandlerSocketBase)
  private
    FTLSServer: TsmCustomTLSInternalServer;
    FTLSSocket: TCustomTLSIdSock;
    FCertificateError: Boolean;
    FSSLStarted: Boolean;
    function GetPassThrough: Boolean;
    procedure SetTLSServer(const Value: TsmCustomTLSInternalServer);
  protected
    procedure DetachBinding;

    procedure InitComponent; override;

    procedure ConnectClient; override;
    procedure SetPassThrough(const Value: Boolean); override;
    function RecvEnc(var ABuffer: TBytes): Integer; override;
    function SendEnc(const ABuffer: TBytes; const AOffset: Integer;
      const ALength: Integer): Integer; override;
  public
    destructor Destroy; override;

    function Clone :  TIdSSLIOHandlerSocketBase; override;
    procedure StartSSL; override;

    procedure AfterAccept; override;
    procedure Close; override;
    procedure Open; override;

    function Recv(var ABuf : TIdBytes): integer;
    function Send(const ABuf : TIdBytes): integer;

    property PassThrough: Boolean read GetPassThrough write SetPassThrough;
    property TLSSocket: TCustomTLSIdSock read FTLSSocket;
    property CertificateError: Boolean read FCertificateError;
  published
    property TLSServer: TsmCustomTLSInternalServer read FTLSServer write SetTLSServer;
  end;

  TsmTLSIdServerIOHandler = class(TIdServerIOHandlerSSLBase)
  private
    FTLSServer: TsmCustomTLSInternalServer;
    FIsInitialized: Boolean;
    FDefaultToPassthrough: Boolean;
    procedure SetTLSServer(const Value: TsmCustomTLSInternalServer);
    procedure SetDefaultToPassthrough(const Value: Boolean);
  protected
    procedure InitComponent; override;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  public
    procedure Init; override;
    function Accept(ASocket: TIdSocketHandle;
                    AListenerThread: TIdThread;
                    AYarn: TIdYarn): TIdIOHandler; override;
    function MakeClientIOHandler : TIdSSLIOHandlerSocketBase; override;

    //
    function MakeFTPSvrPort : TIdSSLIOHandlerSocketBase; override;
    function MakeFTPSvrPasv : TIdSSLIOHandlerSocketBase; override;
    destructor Destroy; override;
  published
    property TLSServer: TsmCustomTLSInternalServer read FTLSServer write SetTLSServer;
    property DefaultToPassthrough: Boolean read FDefaultToPassthrough write SetDefaultToPassthrough;
  end;

implementation

uses
  IdResourceStrings, IdResourceStringsCore, IdException, IdExceptionCore,
  IdStackBSDBase, IdAntiFreezeBase,
  stDERUtils,
  StreamSec.Mobile.TlsConst;

{ TSsTLSInIOHandlerSocket }

procedure TsmTLSIdIOHandlerSocket.AfterAccept;
begin        
  try
    inherited AfterAccept;
    StartSSL;
  except
    Close;
    raise;
  end;
end;

function TsmTLSIdIOHandlerSocket.Clone: TIdSSLIOHandlerSocketBase;
var
  LIO: TsmTLSIdIOHandlerSocket;
begin
  LIO := TsmTLSIdIOHandlerSocket.Create(nil);
  LIO.TLSServer := TLSServer;
  Result := LIO;
end;

type
  TIdSocketHandleHack = class(TIdSocketHandle);

procedure TsmTLSIdIOHandlerSocket.Close;
begin
  try
    if Assigned(FTLSSocket) then begin
      DetachBinding;
    end;
    inherited Close;
    if BindingAllocated then
      Binding.Reset(True);
    inherited SetPassthrough(False);
  except
    on EIdException do
      ;
  else
  end;
end;

procedure TsmTLSIdIOHandlerSocket.ConnectClient;
begin
(*
  inherited Passthrough := True;
*)
  inherited;
  if Assigned(TLSServer) and Binding.HandleAllocated then begin
(*
    inherited Passthrough := True;
*)
    DetachBinding;
    FSSLStarted := True;
    FTLSSocket := TTLSIdSockSlave.CreateSocket(Binding.Handle);
    FTLSSocket.Host := Host;
    FTLSSocket.Port := Port;
    FTLSSocket.TLSServer := TLSServer;
    FTLSSocket.URIToCheck := URIToCheck;
    FTLSSocket.TimeOut := 20000;
    FTLSSocket.IPToCheck := Binding.PeerIP;
    if vsnDNS in FTLSServer.Options.VerifyServerName then
      FTLSSocket.DNSNameToCheck := Host;
//      FTLSSocket.DNSNameToCheck := GStack.HostByAddress(Binding.PeerIP);
    if not Passthrough then begin
      FTLSSocket.InternalSetConnected(True);
      FTLSSocket.Connect;
      if not FTLSSocket.Connected then begin
        case FTLSSocket.ErrorCode and $FF of
          StreamSec.Mobile.TlsConst.handshake_failure,
          StreamSec.Mobile.TlsConst.bad_certificate,
          StreamSec.Mobile.TlsConst.unsupported_certificate,
          StreamSec.Mobile.TlsConst.certificate_revoked,
          StreamSec.Mobile.TlsConst.certificate_expired,
          StreamSec.Mobile.TlsConst.certificate_unknown,
          StreamSec.Mobile.TlsConst.unknown_ca,
          StreamSec.Mobile.TlsConst.access_denied,
          StreamSec.Mobile.TlsConst.export_restriction,
          StreamSec.Mobile.TlsConst.protocol_version,
          StreamSec.Mobile.TlsConst.insufficient_security:
            FCertificateError := True;
        else
          FCertificateError := False;
        end;
        Close;
      end;
    end else
  end;
end;

destructor TsmTLSIdIOHandlerSocket.Destroy;
begin
  TLSServer := nil;
  inherited;
end;

procedure TsmTLSIdIOHandlerSocket.DetachBinding;
begin
  if Assigned(FTLSSocket) then begin
    if Passthrough then try
      FTLSSocket.InternalSetConnected(False);
      if (FTLSSocket.FSocket = Binding) or (FTLSSocket.FSocket = nil) then
        FTLSSocket.FSocket := nil
      else if Assigned(Binding) and
              (FTLSSocket.FSocket.Handle = Binding.Handle) then
        TIdSocketHandleHack(FTLSSocket.FSocket).SetHandle(Id_INVALID_SOCKET);
    except
    end else if FTLSSocket.Encrypted then
      FTLSSocket.Disconnect;
    FTLSSocket.Free;
    FTLSSocket := nil;
    FSSLStarted := False;
  end;
end;

function TsmTLSIdIOHandlerSocket.GetPassThrough: Boolean;
begin
//  Result := not Assigned(FTLSServer);
  Result := inherited Passthrough or not Assigned(FTLSServer);
end;

procedure TsmTLSIdIOHandlerSocket.InitComponent;
begin
  inherited;
  if TLSServer = nil then
    TLSServer := GlobalServer;
end;

procedure TsmTLSIdIOHandlerSocket.Open;
begin
  inherited Open;
end;

(*
procedure TSsTLSInIOHandlerSocket.OpenEncodedConnection;
begin
  // Not done here.
end;
*)

function TsmTLSIdIOHandlerSocket.Recv(var ABuf: TIdBytes): integer;
begin
  if PassThrough then
    Result := Binding.Receive(ABuf)
  else begin
    Result := RecvEnc(ABuf);
  end;
end;

function TsmTLSIdIOHandlerSocket.RecvEnc(var ABuffer: TBytes): Integer;
var
  DLen: Integer;
begin
  if not FSSLStarted then begin
    StartSSL;
    if not FSSLStarted then
      raise EIdException.Create('Unable to start SSL: No binding allocated');
  end;

  Result := 0;
  DLen := 0;
  while FTLSSocket.Connected and (DLen = 0) do begin
    DLen := FTLSSocket.ReceiveLength;
    if DLen > 0 then begin
      SetLength(ABuffer,DLen);
      Result := FTLSSocket.ReceiveBuf(ABuffer[0],DLen);
    end;
  end;
end;

function TsmTLSIdIOHandlerSocket.Send(const ABuf: TIdBytes): integer;
begin
  if PassThrough then
    Result := Binding.Send(ABuf,0)
  else begin
    Result := SendEnc(ABuf,0,Length(ABuf));
  end;
end;

function TsmTLSIdIOHandlerSocket.SendEnc(const ABuffer: TBytes; const AOffset,
  ALength: Integer): Integer;
begin
  if not FSSLStarted then begin
    StartSSL;
    if not FSSLStarted then
      raise EIdException.Create('Unable to start SSL: No binding allocated');
  end;

  if not Assigned(FTLSSocket) then
    raise EIdException.Create('SSL/TLS connection closed');
  Result := ALength;
  if Result > 0 then
    FTLSSocket.SendBuf(ABuffer[AOffset],Result,True);
end;

procedure TsmTLSIdIOHandlerSocket.SetPassThrough(const Value: Boolean);
begin
  Assert(Value or Assigned(FTLSServer));
  if Passthrough and not Value then
    StartSSL;
  inherited SetPassThrough(Value);
end;

procedure TsmTLSIdIOHandlerSocket.SetTLSServer(
  const Value: TsmCustomTLSInternalServer);
begin
  FTLSServer := Value;
  if Value = nil then
    Passthrough := True;
end;

{ TCustomTLSIdSock }

constructor TCustomTLSIdSock.Create(ASocket: TIdSocketHandle);
begin
  inherited Create;
  FSocket := ASocket;
  if Assigned(ASocket) then begin
    FBindAddress := ASocket.IP;
    if FBindAddress = '' then
      FBindAddress := '0.0.0.0';
    FBindMaxPort := ASocket.ClientPortMax;
    if FBindMaxPort = 0 then
      FBindMaxPort := 65535;
    FTimeOut := 20000;
  end;
end;

procedure TCustomTLSIdSock.DoRead;
begin
  // Implemented in descendant
end;

function TCustomTLSIdSock.GetRemoteAddress: string;
begin
  Result := FRemoteAddress;
  if Result = '' then begin
    Result := FSocket.PeerIP;
    FRemoteAddress := Result;
  end;
end;

function TCustomTLSIdSock.GetRemoteHost: string;
begin
  Result := FRemoteHost;
  if Result = '' then begin
    Result := FSocket.PeerIP;
    FRemoteHost := Result;
  end;
end;

function TCustomTLSIdSock.GetRemotePort: Integer;
begin
  Result := FRemotePort;
  if Result = 0 then begin
    Result := FSocket.PeerPort;
    FRemotePort := Result;
  end;
end;

procedure TCustomTLSIdSock.RawClose;
begin
  if Assigned(FSocket) then
    FSocket.CloseSocket;
end;

procedure TCustomTLSIdSock.RawConnect;
begin
  if not FSocket.HandleAllocated then begin
    if FHost = '' then
      FHost := FAddress;
    FSocket.IP := FBindAddress;
    FSocket.Port := FBindPort;
    FSocket.ClientPortMin := FBindMinPort;
    FSocket.ClientPortMax := FBindMaxPort;
    FSocket.AllocateSocket;
    FSocket.Bind;
  end;
  if GStack.IsIP(FHost) then
    FAddress := FHost
  else
    FAddress := GStack.ResolveHost(FHost);
  if (FSocket.PeerIP <> FAddress) or (FSocket.PeerPort <> FPort) or
     not Connected then begin
    FSocket.SetPeer(FAddress,FPort);
    try
      InternalSetConnected(True);
    except
      InternalSetConnected(False);
      raise;
    end;
  end;
end;

function TCustomTLSIdSock.RawReceive: TStream;
begin
  if FSocket.Readable(FTimeOut) then begin
    Result := TInSocketStream.Create(FSocket,FTimeOut);
  end else
    Result := nil;
end;

procedure TCustomTLSIdSock.RawSend(Strm: TCustomMemoryStream);
var
  lBuffer: TIdBytes;
begin
  SetLength(lBuffer,Strm.Size);
  Move(Strm.Memory^,Pointer(lBuffer)^,Strm.Size);
  if FSocket.Send(lBuffer,0,Length(lBuffer)) <> Length(lBuffer) then
    InternalSetConnected(False);
end;

procedure TCustomTLSIdSock.SetAddress(const Value: string);
begin
  FAddress := Value;
end;

procedure TCustomTLSIdSock.SetBindAddress(const Value: string);
begin
  FBindAddress := Value;
end;

procedure TCustomTLSIdSock.SetBindMaxPort(const Value: Integer);
begin
  FBindMaxPort := Value;
end;

procedure TCustomTLSIdSock.SetBindMinPort(const Value: Integer);
begin
  FBindMinPort := Value;
end;

procedure TCustomTLSIdSock.SetBindPort(const Value: Integer);
begin
  FBindPort := Value;
end;

procedure TCustomTLSIdSock.SetHost(const Value: string);
begin
  FHost := Value;
end;

procedure TCustomTLSIdSock.SetPort(const Value: Integer);
begin
  FPort := Value;
end;

procedure TCustomTLSIdSock.SetService(const Value: string);
begin
  FService := Value;
end;

procedure TCustomTLSIdSock.SetTimeOut(const Value: Cardinal);
begin
  FTimeOut := Value;
end;

{ TTLSIdSockSlave }

procedure TTLSIdSockSlave.Connect;
begin
  InternalConnect;
  while Connected and not Encrypted do
    InternalRead;
end;

constructor TTLSIdSockSlave.Create(ASocket: TIdSocketHandle);
begin
  inherited;
end;

constructor TTLSIdSockSlave.CreateSocket(ASocket: TIdStackSocketHandle);
begin
  FSlaveSocket := TIdSocketHandle.Create(nil);
  with TIdSocketHandleHack(FSlaveSocket) do begin
    SetHandle(ASocket);
    FHandleAllocated := ASocket <> Id_INVALID_SOCKET;
  end;
  InternalSetConnected(ASocket <> Id_INVALID_SOCKET);
  inherited Create(FSlaveSocket);
end;

destructor TTLSIdSockSlave.Destroy;
begin
  try
    InternalDisconnect;
  except
  end;
  FSlaveSocket.Free;
  inherited;
end;

procedure TTLSIdSockSlave.Disconnect;
begin
  InternalDisconnect;
end;

procedure TTLSIdSockSlave.DoConnect;
begin
  DoSleep(0);
  GetRemoteHost;
  InternalSetConnected(FSocket.Handle <> Id_INVALID_SOCKET);
  while Connected and not Encrypted do begin
    DoSleep(1);
    Receive;
  end;
  DoSleep(0);
end;

procedure TTLSIdSockSlave.Receive;
begin
  InternalRead;
  DoSleep(0);
end;

procedure TTLSIdSockSlave.Send;
begin
  InternalSend;
end;

{ TBlockTLSIdSock }

procedure TBlockTLSIdSock.DoConnect;
begin
  inherited;
  if Connected and Encrypted then
    if Assigned(FOnConnect) then
      FOnConnect(Self);
end;

procedure TBlockTLSIdSock.SetOnConnect(const Value: TNotifyEvent);
begin
  FOnConnect := Value;
end;

{ TIdSocketStream }

constructor TInSocketStream.Create(ASocket: TIdSocketHandle;
  TimeOut: Integer);
begin
  FSocket := ASocket;
  if TimeOut <= 0 then TimeOut := 20000;
  FTimeOut := TimeOut;
end;

function TInSocketStream.Read(var Buffer; Count: Integer): Longint;
var
  LBuffer: TIdBytes;
begin
  if Count > 0 then begin
    if WaitForData(FTimeOut) then begin
      SetLength(LBuffer,Count);
      Result := FSocket.Receive(LBuffer);
      Move(LBuffer[0],Buffer,Result);
    end else
      Result := 0;
    FReadable := False;
  end else
    Result := 0;
end;

function TInSocketStream.Seek(Offset: Integer; Origin: Word): Longint;
begin
  Result := 0;
end;

function TInSocketStream.WaitForData(Timeout: Integer): Boolean;
begin
  Result := FReadable;
  if not Result then begin
    Result := FSocket.Readable(Timeout);
    FReadable := Result;
  end;
end;

function TInSocketStream.Write(const Buffer; Count: Integer): Longint;
var
  lBuf: TIdBytes;
begin
  lBuf := TIdBytes(CopyUntypedToBytes(Buffer,Count));
  Result := FSocket.Send(lBuf,0,Count);
end;

{ TSsTLSIdServerIOHandler }

function TsmTLSIdServerIOHandler.Accept(ASocket: TIdSocketHandle;
  AListenerThread: TIdThread; AYarn: TIdYarn): TIdIOHandler;
var
  tmpInCISsTLS: TsmTLSIdIOHandlerSocket;
begin
  if not FIsInitialized then begin
    Init;
  end;

  Result := nil;
  tmpInCISsTLS := TsmTLSIdIOHandlerSocket.Create(nil);
  try
    tmpInCISsTLS.PassThrough := true;
    tmpInCISsTLS.fIsPeer := True;
    tmpInCISsTLS.Open;
    if tmpInCISsTLS.Binding.Accept(ASocket.Handle) then begin
      tmpInCISsTLS.TLSServer := FTLSServer;
      if not FDefaultToPassthrough then
        tmpInCISsTLS.AfterAccept;
      Result := tmpInCISsTLS;
      tmpInCISsTLS := nil;
    end;
  finally
    tmpInCISsTLS.Free;
  end;
end;

destructor TsmTLSIdServerIOHandler.Destroy;
begin
  inherited;
end;

procedure TsmTLSIdServerIOHandler.Init;
begin
  if FTLSServer = nil then
    FTLSServer := GlobalServer;
  if FTLSServer = nil then
    raise Exception.Create('TSsTLSIdServerIOHandler.Init: No TLSServer assigned');
  if FTLSServer.PublicKeyAlgorithms = [] then
    raise Exception.Create('TSsTLSIdServerIOHandler.Init: No Server Certificate');
  FTLSServer.TLSSetupServer;
  FIsInitialized := True;
end;

procedure TsmTLSIdServerIOHandler.InitComponent;
begin
  inherited;
  FIsInitialized := False;
  TLSServer := GlobalServer;
end;

function TsmTLSIdServerIOHandler.MakeClientIOHandler: TIdSSLIOHandlerSocketBase;
var
  LIO: TsmTLSIdIOHandlerSocket;
begin
  LIO := TsmTLSIdIOHandlerSocket.Create(nil);
  LIO.TLSServer := TLSServer;
  Result := LIO;
end;

function TsmTLSIdServerIOHandler.MakeFTPSvrPasv: TIdSSLIOHandlerSocketBase;
var
  LIO: TsmTLSIdIOHandlerSocket;
begin
  LIO := TsmTLSIdIOHandlerSocket.Create(nil);
  LIO.TLSServer := TLSServer;
  LIO.IsPeer := True;
  Result := LIO;
end;

function TsmTLSIdServerIOHandler.MakeFTPSvrPort: TIdSSLIOHandlerSocketBase;
var
  LIO: TsmTLSIdIOHandlerSocket;
begin
  LIO := TsmTLSIdIOHandlerSocket.Create(nil);
  LIO.TLSServer := TLSServer;
  LIO.IsPeer := True;
  Result := LIO;
end;

procedure TsmTLSIdServerIOHandler.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited;
  if Operation = opRemove then begin
    if AComponent = FTLSServer then
      TLSServer := nil;
  end;
end;

procedure TsmTLSIdServerIOHandler.SetDefaultToPassthrough(const Value: Boolean);
begin
  FDefaultToPassthrough := Value;
end;

procedure TsmTLSIdServerIOHandler.SetTLSServer(
  const Value: TsmCustomTLSInternalServer);
begin
  if Value <> FTLSServer then begin
    FTLSServer := Value;
    if Assigned(Value) then
      Value.FreeNotification(Self)
    else
      FIsInitialized := False;
  end;
end;

procedure TsmTLSIdIOHandlerSocket.StartSSL;
begin
  if Assigned(FTLSServer) then begin
    DetachBinding;
    if Binding = nil then begin
      FSSLStarted := False;
    end else begin
      FSSLStarted := True;
      FTLSSocket := TTLSIdSockSlave.Create(Binding);
      FTLSSocket.TLSServer := TLSServer;
      FTLSSocket.TimeOut := 20000;
      FTLSSocket.URIToCheck := URIToCheck;
      if IsPeer then begin
        TTLSIdSockSlave(FTLSSocket).DoConnect;
        if not FTLSSocket.Connected then
          Close;
      end else begin
        FTLSSocket.Address := Host;
        FTLSSocket.Port := Port;
        if Assigned(Binding) then begin
          FTLSSocket.IPToCheck := Binding.PeerIP;
          FTLSSocket.DNSNameToCheck := GStack.HostByAddress(Binding.IP);
        end;
        FTLSSocket.Connect;
      end;
    end;
    inherited SetPassthrough(False);
  end;
end;

initialization
  RegisterSSL('StreamSec Tools 4.0 for IntraWeb',
    'StreamSec',
    'Copyright  2000 - 2015'#10#13 +                                     {do not localize}
    'StreamSec Handelsbolag. All rights reserved.',                       {do not localize}
    'StreamSec TLS for IntraWeb 7,1',                                     {do not localize}
    'http://www.streamsec.com/',                                          {do not localize}
    TsmTLSIdIOHandlerSocket,
    TsmTLSIdServerIOHandler);
end.
